const IMAGE_MAX = 3;

function onClassChange(element, callback) {
  const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      if (
        mutation.type === "attributes" &&
        mutation.attributeName === "class"
      ) {
        callback(mutation.target);
      }
    });
  });
  observer.observe(element, { attributes: true });
  return observer.disconnect;
}

function nextSlide() {
	const lightBoxImages = document.querySelectorAll('.lightbox .image-container div');
	const currentImageIndex = Array.from(lightBoxImages).findIndex(node => node.classList.contains("current"));
	setCurrentImage((currentImageIndex + 1) % lightBoxImages.length);
	updateImageCount(lightBoxImages.length, (currentImageIndex + 1) % (lightBoxImages.length));
}

function prevSlide() {
	const lightBoxImages = document.querySelectorAll('.lightbox .image-container div');
	const currentImageIndex = Array.from(lightBoxImages).findIndex(node => node.classList.contains("current"));
	if (currentImageIndex < 1) {
		setCurrentImage(lightBoxImages.length - 1);
		updateImageCount(lightBoxImages.length, lightBoxImages.length - 1);
		return;
	}
	setCurrentImage( (currentImageIndex - 1) % lightBoxImages.length);
	updateImageCount(lightBoxImages.length, (currentImageIndex - 1) % lightBoxImages.length);
}

function initLightBox(){
	const lightBoxContainer = document.createElement('div');
	lightBoxContainer.classList.add('lightbox');
	
	const navigationArrows = () => {
		const navContainer = document.createElement('div');
		navContainer.classList.add('pagination');
		
		const nextArrow = document.createElement('div');
		nextArrow.addEventListener('click', () => {
			nextSlide();
		});
		const prevArrow = document.createElement('div');
		prevArrow.addEventListener('click', ()=> {
			prevSlide();
		});
		
		// prevArrow.innerHTML = '<i data-icon=""></i>';
		// nextArrow.innerHTML = '<i data-icon=""></i>';
		
		prevArrow.innerHTML = '&#5176;';
		nextArrow.innerHTML = '&#5171;';
		
		navContainer.appendChild(prevArrow);
		navContainer.appendChild(nextArrow);
		return navContainer;
	}
	
	const closeButton = () => {
		const closeButtonContainer = document.createElement("div");
		closeButtonContainer.classList.add('close-btn');
		closeButtonContainer.innerHTML = "&#65794;";
		
		closeButtonContainer.addEventListener('click', () => {
			lightBoxContainer.classList.remove('open');
		});
		return closeButtonContainer;
	}
	
	const imageCount = () => {
		const imageCountContainer = document.createElement("div");
		imageCountContainer.classList.add('slide-count');
		imageCountContainer.innerHTML = "/";
		
		return imageCountContainer;
	}
	
	const imagesContainer = () => {
		const imagesContainer = document.createElement("div");
		imagesContainer.classList.add("image-container");
		
		return imagesContainer;
	}
	
	lightBoxContainer.appendChild(imagesContainer());
	lightBoxContainer.appendChild(imageCount());
	lightBoxContainer.appendChild(closeButton());
	lightBoxContainer.appendChild(navigationArrows());
	
	document.body.appendChild(lightBoxContainer);
}

function setLightBoxListener(lightBoxDataElement) {
	const lightBoxImages = lightBoxDataElement.querySelectorAll(":scope img");
	if (lightBoxImages.length <= IMAGE_MAX) return;
	
	Array.from(lightBoxDataElement
		.querySelectorAll(":scope > *"))
		.forEach((imageElement, index) => {
			imageElement.addEventListener('click', () => {
				clearLightBoxImages();
				setLightBoxImages(lightBoxImages);
				openLightBox(index);
			});
	});
	
	onClassChange(document.querySelector(".lightbox"), node => {
		if( node.classList.contains('open')) toggleScrollLock(true);
		else toggleScrollLock();
	});
}

function toggleScrollLock(lock = false) {
	if (lock){
		document.documentElement.style.overflow = 'hidden';
		document.documentElement.onscroll = function() {
			window.scrollTo(0,0);
		}
	} else {
		document.documentElement.style = '';
		document.documentElement.onscroll = function() {};
	}
}

function setCurrentImage(index = 0) {
	const lightBoxImages = document.querySelectorAll(".lightbox .image-container div");
	Array
		.from(lightBoxImages)
		.forEach((image, ind) => {
			image.classList.toggle("current", index === ind);
			image.style.transform = `translateX(${(ind - index) * 120}%)`;
	});
}

function openLightBox(index = 0) {
	const lightBoxImages = document.querySelectorAll(".lightbox .image-container div");
	setCurrentImage(index);
	document.querySelector('.lightbox').classList.toggle("open");
	updateImageCount(lightBoxImages.length, index);
}

function setLightBoxImages(imagesElement) {
	const lightBox = document.querySelector(".lightbox .image-container");
	
	Array.from(imagesElement)
		.forEach((image, index) => {
			const imageContainer = document.createElement('div');
		
			image.removeAttribute("width");
			image.removeAttribute("height");
		
			imageContainer.appendChild(image.cloneNode(true));
			imageContainer.style.transform = `translateX(${(index) * 100}%)`;
			lightBox.appendChild(imageContainer);
	});
	updateImageCount(imagesElement.length);
}

function clearLightBoxImages() {
	const lightBox = document.querySelector(".lightbox .image-container");
	lightBox.innerHTML = "";
}

function updateImageCount(total, index = 0) {
	document.querySelector(".slide-count").innerHTML = `${index + 1} / ${total}`;
}

window.addEventListener('load', () => {
	const lightboxContainer = document.querySelectorAll(".lightbox-container");
	if (lightboxContainer != null) {
		initLightBox();
		lightboxContainer.forEach(lightbox => {
			setLightBoxListener(lightbox);
		});
	}	
});