<?php

/**
 * Settings page controller
 *
 * @package   Duplicator
 * @copyright (c) 2022, Snap Creek LLC
 */

namespace Duplicator\Controllers;

use DUP_PRO_Constants;
use DUP_PRO_DB;
use DUP_PRO_Global_Entity;
use DUP_PRO_Log;
use DUP_PRO_Secure_Global_Entity;
use DUP_PRO_Server_Load_Reduction;
use DUP_PRO_U;
use Duplicator\Core\CapMng;
use Duplicator\Core\Controllers\ControllersManager;
use Duplicator\Core\Controllers\AbstractMenuPageController;
use Duplicator\Core\Controllers\PageAction;
use Duplicator\Core\Controllers\SubMenuItem;
use Duplicator\Core\Views\TplMng;
use Duplicator\Libs\Snap\SnapUtil;
use Duplicator\Models\DynamicGlobalEntity;

class SettingsPageController extends AbstractMenuPageController
{
    const NONCE_ACTION = 'duppro-settings-package';

    /**
     * tabs menu
     */
    const L2_SLUG_GENERAL         = 'general';
    const L2_SLUG_GENERAL_MIGRATE = 'migrate';
    const L2_SLUG_PACKAGE_BRAND   = 'brand';
    const L2_SLUG_PACKAGE         = 'package';
    const L2_SLUG_SCHEDULE        = 'schedule';
    const L2_SLUG_STORAGE         = 'storage';
    const L2_SLUG_IMPORT          = 'import';
    const L2_SLUG_CAPABILITIES    = 'capabilities';

    const BRAND_INNER_PAGE_LIST = 'list';
    const BRAND_INNER_PAGE_EDIT = 'edit';

    /*
     * action types
     */
    const ACTION_GENERAL_SAVE          = 'save';
    const ACTION_GENERAL_TRACE         = 'trace';
    const ACTION_CAPABILITIES_SAVE     = 'cap-save';
    const ACTION_CAPABILITIES_RESET    = 'cap-reset';
    const ACTION_IMPORT_SAVE_SETTINGS  = 'import-save-set';
    const ACTION_PACKAGE_ADVANCED_SAVE = 'pack-adv-save';
    const ACTION_PACKAGE_BASIC_SAVE    = 'pack-basic-save';
    const ACTION_RESET_SETTINGS        = 'reset-settings';
    const ACTION_SAVE_STORAGE          = 'save-storage';
    const ACTION_SAVE_STORAGE_SSL      = 'save-storage-ssl';
    const ACTION_SAVE_STORAGE_OPTIONS  = 'save-storage-options';

    /**
     * Class constructor
     */
    protected function __construct()
    {
        $this->parentSlug   = ControllersManager::MAIN_MENU_SLUG;
        $this->pageSlug     = ControllersManager::SETTINGS_SUBMENU_SLUG;
        $this->pageTitle    = __('Settings', 'duplicator-pro');
        $this->menuLabel    = __('Settings', 'duplicator-pro');
        $this->capatibility = CapMng::CAP_SETTINGS;
        $this->menuPos      = 60;

        add_filter('duplicator_sub_menu_items_' . $this->pageSlug, array($this, 'getBasicSubMenus'));
        add_filter('duplicator_sub_level_default_tab_' . $this->pageSlug, array($this, 'getSubMenuDefaults'), 10, 2);
        add_action('duplicator_render_page_content_' . $this->pageSlug, array($this, 'renderContent'), 10, 2);
        add_filter('duplicator_page_actions_' . $this->pageSlug, array($this, 'pageActions'));
    }

    /**
     * Return sub menus for current page
     *
     * @param SubMenuItem[] $subMenus sub menus list
     *
     * @return SubMenuItem[]
     */
    public function getBasicSubMenus($subMenus)
    {
        $subMenus[] = new SubMenuItem(self::L2_SLUG_GENERAL, __('General', 'duplicator-pro'));
        $subMenus[] = new SubMenuItem(self::L2_SLUG_PACKAGE, __('Backups', 'duplicator-pro'));
        $subMenus[] = new SubMenuItem(self::L2_SLUG_PACKAGE_BRAND, __('Installer Branding', 'duplicator-pro'));
        $subMenus[] = new SubMenuItem(self::L2_SLUG_SCHEDULE, __('Schedules', 'duplicator-pro'));
        $subMenus[] = new SubMenuItem(self::L2_SLUG_STORAGE, __('Storage', 'duplicator-pro'));
        $subMenus[] = new SubMenuItem(self::L2_SLUG_IMPORT, __('Import', 'duplicator-pro'));
        $subMenus[] = new SubMenuItem(self::L2_SLUG_GENERAL_MIGRATE, __('Migrate Settings', 'duplicator-pro'));
        $subMenus[] = new SubMenuItem(self::L2_SLUG_CAPABILITIES, __('Access', 'duplicator-pro'));

        return $subMenus;
    }

    /**
     * Return slug default for parent menu slug
     *
     * @param string $slug   current default
     * @param string $parent parent for default
     *
     * @return string default slug
     */
    public function getSubMenuDefaults($slug, $parent)
    {
        switch ($parent) {
            case '':
                return self::L2_SLUG_GENERAL;
            default:
                return $slug;
        }
    }

    /**
     * Return actions for current page
     *
     * @param PageAction[] $actions actions lists
     *
     * @return PageAction[]
     */
    public function pageActions($actions)
    {
        $actions[] = new PageAction(
            self::ACTION_GENERAL_SAVE,
            array(
                $this,
                'saveGeneral',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_GENERAL,
            )
        );
        $actions[] = new PageAction(
            self::ACTION_GENERAL_TRACE,
            array(
                $this,
                'traceGeneral',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_GENERAL,
            )
        );
        $actions[] = new PageAction(
            self::ACTION_CAPABILITIES_SAVE,
            array(
                $this,
                'saveCapabilities',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_CAPABILITIES,
            )
        );
        $actions[] = new PageAction(
            self::ACTION_CAPABILITIES_RESET,
            array(
                $this,
                'resetCapabilities',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_CAPABILITIES,
            )
        );
        $actions[] = new PageAction(
            self::ACTION_PACKAGE_BASIC_SAVE,
            array(
                $this,
                'savePackage',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_PACKAGE,
            )
        );
        $actions[] = new PageAction(
            self::ACTION_IMPORT_SAVE_SETTINGS,
            array(
                $this,
                'saveImportSettngs',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_IMPORT,
            )
        );
        $actions[] = new PageAction(
            self::ACTION_RESET_SETTINGS,
            array(
                $this,
                'resetSettings',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_GENERAL,
            )
        );
        $actions[] = new PageAction(
            self::ACTION_SAVE_STORAGE,
            array(
                $this,
                'saveStorageGeneral',
            ),
            array(
                $this->pageSlug,
                self::L2_SLUG_STORAGE,
            )
        );

        return $actions;
    }


    /**
     * Render page content
     *
     * @param string[] $currentLevelSlugs current menu slugs
     * @param string   $innerPage         current inner page, empty if not set
     *
     * @return void
     */
    public function renderContent($currentLevelSlugs, $innerPage)
    {
        switch ($currentLevelSlugs[1]) {
            case self::L2_SLUG_GENERAL:
                TplMng::getInstance()->render('admin_pages/settings/general/general');
                break;
            case self::L2_SLUG_PACKAGE_BRAND:
                switch ($innerPage) {
                    case self::BRAND_INNER_PAGE_EDIT:
                        TplMng::getInstance()->render('admin_pages/settings/brand/brand_edit');
                        break;
                    case self::BRAND_INNER_PAGE_LIST:
                    default:
                        TplMng::getInstance()->render('admin_pages/settings/brand/brand_list');
                        break;
                }
                break;
            case self::L2_SLUG_GENERAL_MIGRATE:
                TplMng::getInstance()->render('admin_pages/settings/migrate_settings/migrate_page');
                break;
            case self::L2_SLUG_PACKAGE:
                TplMng::getInstance()->render('admin_pages/settings/backup/backup_settings');
                break;
            case self::L2_SLUG_IMPORT:
                TplMng::getInstance()->render('admin_pages/settings/import/import');
                break;
            case self::L2_SLUG_SCHEDULE:
                TplMng::getInstance()->render('admin_pages/settings/schedule/schedule');
                break;
            case self::L2_SLUG_STORAGE:
                TplMng::getInstance()->render('admin_pages/settings/storage/storage_settings');
                break;
            case self::L2_SLUG_CAPABILITIES:
                TplMng::getInstance()->render('admin_pages/settings/capabilities/capabilites');
                break;
        }
    }

    /**
     * Save general settings
     *
     * @return array<string, mixed>
     */
    public function saveGeneral()
    {
        $result = ['saveSuccess' => false];
        $global = DUP_PRO_Global_Entity::getInstance();

        $global->uninstall_settings = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'uninstall_settings');
        $global->uninstall_packages = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'uninstall_packages');

        $cryptSettingChanged = (SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'crypt') != $global->isEncryptionEnabled());

        if ($cryptSettingChanged) {
            do_action('duplicator_before_update_crypt_setting');
        }

        $global->setEncryption(SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'crypt'));
        $global->unhook_third_party_js  = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, '_unhook_third_party_js');
        $global->unhook_third_party_css = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, '_unhook_third_party_css');

        $this->updateLoggingModeOptions();

        $global->setEmailSummaryFrequency(SnapUtil::sanitizeTextInput(SnapUtil::INPUT_REQUEST, '_email_summary_frequency'));
        $emailRecipients = filter_input(INPUT_POST, '_email_summary_recipients', FILTER_SANITIZE_EMAIL, [
            'flags'   => FILTER_REQUIRE_ARRAY,
            'options' => [
                'default' => [],
            ],
        ]);
        if ($emailRecipients !== []) {
            $emailRecipients = array_map('sanitize_email', $emailRecipients);
        }
        $global->setEmailSummaryRecipients($emailRecipients);
        $global->setUsageTracking(SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'usage_tracking'));
        $global->setAmNotices(!SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'dup_am_notices'));

        if (($result['saveSuccess'] = $global->save()) == false) {
            $result['errorMessage'] = __('Can\'t update general settings', 'duplicator-pro');
            return $result;
        } else {
            $result['successMessage'] = __("General settings updated.", 'duplicator-pro');
        }

        if ($cryptSettingChanged) {
            do_action('duplicator_after_update_crypt_setting');
        }

        return $result;
    }

    /**
     * Save capabilities settings
     *
     * @return array<string, mixed>
     */
    public function saveCapabilities()
    {
        $result = ['saveSuccess' => false];

        $capabilities = [];
        foreach (CapMng::getCapsList() as $capName) {
            $capabilities[$capName] = [
                'roles' => [],
                'users' => [],
            ];

            $inputName = TplMng::getInputName('cap', $capName);
            $result    = filter_input(INPUT_POST, $inputName, FILTER_UNSAFE_RAW, [
                'flags'   => FILTER_REQUIRE_ARRAY,
                'options' => [
                    'default' => [],
                ],
            ]);
            if ($result === []) {
                continue;
            }

            foreach ($result as $roles) {
                $roles = SnapUtil::sanitizeNSCharsNewlineTrim($roles);
                if (is_numeric($roles)) {
                    $capabilities[$capName]['users'][] = (int) $roles;
                } else {
                    $capabilities[$capName]['roles'][] = $roles;
                }
            }
        }

        if (CapMng::getInstance()->update($capabilities) == false) {
            $result['saveSuccess']  = false;
            $result['errorMessage'] = __('Can\'t update capabilities.', 'duplicator-pro');
            return $result;
        } else {
            $result['successMessage'] = __('Capabilities updated.', 'duplicator-pro');
            $result['saveSuccess']    = true;
        }

        return $result;
    }

    /**
     * Reset capabilities settings
     *
     * @return array<string, mixed>
     */
    public function resetCapabilities()
    {
        $result = ['saveSuccess' => false];

        $capabilities = CapMng::getDefaultCaps();
        if (!CapMng::can(CapMng::CAP_LICENSE)) {
            // Can't reset license capability if current user can't manage license
            unset($capabilities[CapMng::CAP_LICENSE]);
        }

        if (CapMng::getInstance()->update($capabilities) == false) {
            $result['saveSuccess']  = false;
            $result['errorMessage'] = __('Can\'t update capabilities.', 'duplicator-pro');
        } else {
            $result['successMessage'] = __('Capabilities updated.', 'duplicator-pro');
            $result['saveSuccess']    = true;
        }

        return $result;
    }

    /**
     * Save storage general settings
     *
     * @return array<string, mixed>
     */
    public function saveStorageGeneral()
    {
        $result = ['saveSuccess' => false];

        $global                       = DUP_PRO_Global_Entity::getInstance();
        $global->storage_htaccess_off = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, '_storage_htaccess_off');
        $global->max_storage_retries  = SnapUtil::sanitizeIntInput(SnapUtil::INPUT_REQUEST, 'max_storage_retries', 10);
        $global->ssl_useservercerts   = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'ssl_useservercerts');
        $global->ssl_disableverify    = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'ssl_disableverify');
        $global->ipv4_only            = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, 'ipv4_only');

        if (($result['saveSuccess'] = $global->save()) == false) {
            $result['errorMessage'] = __('Can\'t update storage settings.', 'duplicator-pro');
        } else {
            $result['successMessage'] = __('Storage settings updated.', 'duplicator-pro');
        }

        if ($result['saveSuccess']) {
            do_action('duplicator_update_global_storage_settings');

            $dGlobal = DynamicGlobalEntity::getInstance();
            if (($result['saveSuccess'] = $dGlobal->save()) == false) {
                $result['errorMessage'] = __('Can\'t update storage settings.', 'duplicator-pro');
            } else {
                $result['successMessage'] = __('Storage settings updated.', 'duplicator-pro');
            }
        }

        return $result;
    }

    /**
     * Reset all user settings and redirects to the settings page
     *
     * @return array<string, mixed>
     */
    public function resetSettings()
    {
        $result = ['saveSuccess' => false];

        $global = DUP_PRO_Global_Entity::getInstance();
        if ($global->resetUserSettings() && $global->save()) {
            $result['successMessage'] = __('Settings reset to defaults successfully', 'duplicator-pro');
            $result['saveSuccess']    = true;
        } else {
            $result['errorMessage'] = __('Failed to reset settings.', 'duplicator-pro');
            $result['saveSuccess']  = false;
        }

        return $result;
    }

    /**
     * Update trace mode
     *
     * @return array<string, mixed>
     */
    public function traceGeneral()
    {
        $result = ['saveSuccess' => false];

        switch (SnapUtil::sanitizeStrictInput(SnapUtil::INPUT_REQUEST, '_logging_mode')) {
            case 'off':
                $this->updateLoggingModeOptions();
                $result = [
                    'saveSuccess'    => true,
                    'successMessage' => __("Trace settings have been turned off.", 'duplicator-pro'),
                ];
                break;
            case 'on':
                $this->updateLoggingModeOptions();
                $result = [
                    'saveSuccess'    => true,
                    'successMessage' => __("Trace settings have been turned on.", 'duplicator-pro'),
                ];
                break;
            default:
                $result = [
                    'saveSuccess'  => false,
                    'errorMessage' => __("Trace mode not valid.", 'duplicator-pro'),
                ];
                break;
        }

        return $result;
    }

    /**
     * Upate loggin modes options
     *
     * @return void
     */
    protected function updateLoggingModeOptions()
    {
        switch (SnapUtil::sanitizeStrictInput(SnapUtil::INPUT_REQUEST, '_logging_mode')) {
            case 'off':
                update_option('duplicator_pro_trace_log_enabled', false, true);
                update_option('duplicator_pro_send_trace_to_error_log', false);
                break;
            case 'on':
                if ((bool) get_option('duplicator_pro_trace_log_enabled') == false) {
                    DUP_PRO_Log::deleteTraceLog();
                }
                update_option('duplicator_pro_trace_log_enabled', true, true);
                update_option('duplicator_pro_send_trace_to_error_log', false);
                break;
            case 'enhanced':
                if (
                    ((bool) get_option('duplicator_pro_trace_log_enabled') == false) ||
                    ((bool) get_option('duplicator_pro_send_trace_to_error_log') == false)
                ) {
                    DUP_PRO_Log::deleteTraceLog();
                }

                update_option('duplicator_pro_trace_log_enabled', true, true);
                update_option('duplicator_pro_send_trace_to_error_log', true);
                break;
            default:
                break;
        }
    }



    /**
     * Save Backup basic settings
     *
     * @return array<string, mixed>
     */
    public function savePackage()
    {
        $result          = ['saveSuccess' => false];
        $global          = DUP_PRO_Global_Entity::getInstance();
        $sglobal         = DUP_PRO_Secure_Global_Entity::getInstance();
        $packageBuild    = DUP_PRO_Constants::DEFAULT_MAX_PACKAGE_RUNTIME_IN_MIN;
        $defaultTransfer = DUP_PRO_Constants::DEFAULT_MAX_PACKAGE_TRANSFER_TIME_IN_MIN;

        $global->setDbMode();
        $global->setArchiveMode();
        $global->max_package_runtime_in_min       = SnapUtil::sanitizeIntInput(SnapUtil::INPUT_REQUEST, 'max_package_runtime_in_min', $packageBuild);
        $global->server_load_reduction            = SnapUtil::sanitizeIntInput(SnapUtil::INPUT_REQUEST, 'server_load_reduction', 0);
        $global->max_package_transfer_time_in_min = SnapUtil::sanitizeIntInput(SnapUtil::INPUT_REQUEST, 'max_package_transfer_time_in_min', $defaultTransfer);
        $global->max_package_runtime_in_min       = SnapUtil::sanitizeIntInput(
            INPUT_POST,
            'max_package_runtime_in_min',
            DUP_PRO_Constants::DEFAULT_MAX_PACKAGE_RUNTIME_IN_MIN
        );
        $global->server_load_reduction            = SnapUtil::sanitizeIntInput(
            INPUT_POST,
            'server_load_reduction',
            DUP_PRO_Server_Load_Reduction::None
        );

        switch (SnapUtil::sanitizeDefaultInput(INPUT_POST, 'installer_name_mode')) {
            case DUP_PRO_Global_Entity::INSTALLER_NAME_MODE_WITH_HASH:
                $global->installer_name_mode = DUP_PRO_Global_Entity::INSTALLER_NAME_MODE_WITH_HASH;
                break;
            case DUP_PRO_Global_Entity::INSTALLER_NAME_MODE_SIMPLE:
            default:
                $global->installer_name_mode = DUP_PRO_Global_Entity::INSTALLER_NAME_MODE_SIMPLE;
                break;
        }

        $global->lock_mode       = SnapUtil::sanitizeIntInput(SnapUtil::INPUT_REQUEST, 'lock_mode', 0);
        $global->ajax_protocol   = SnapUtil::sanitizeTextInput(SnapUtil::INPUT_REQUEST, 'ajax_protocol', 'admin');
        $global->custom_ajax_url = SnapUtil::sanitizeTextInput(SnapUtil::INPUT_REQUEST, 'custom_ajax_url', $global->custom_ajax_url);
        $clientSideKickoff       = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, '_clientside_kickoff', false);
        $global->setClientsideKickoff($clientSideKickoff);
        $global->homepath_as_abspath = SnapUtil::sanitizeBoolInput(INPUT_POST, 'homepath_as_abspath', false);

        $global->basic_auth_enabled = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, '_basic_auth_enabled');
        if ($global->basic_auth_enabled == true) {
            $global->basic_auth_user = SnapUtil::sanitizeTextInput(SnapUtil::INPUT_REQUEST, 'basic_auth_user', '');
        } else {
            $global->basic_auth_user     = '';
            $global->basic_auth_password = '';
        }
        $installer_base_name                = SnapUtil::sanitizeTextInput(SnapUtil::INPUT_REQUEST, '_installer_base_name', 'installer.php');
        $global->installer_base_name        = stripslashes($installer_base_name);
        $global->chunk_size                 = SnapUtil::sanitizeIntInput(SnapUtil::INPUT_REQUEST, '_chunk_size', 2048);
        $global->skip_archive_scan          = SnapUtil::sanitizeBoolInput(SnapUtil::INPUT_REQUEST, '_skip_archive_scan', false);
        $global->php_max_worker_time_in_sec = SnapUtil::sanitizeIntInput(
            SnapUtil::INPUT_REQUEST,
            'php_max_worker_time_in_sec',
            DUP_PRO_Constants::DEFAULT_MAX_WORKER_TIME
        );

        // CLEANUP
        $global->setCleanupFields();

        if (($result['saveSuccess'] = $global->save()) == false) {
            $result['errorMessage'] = __('Can\'t Save Backup Settings', 'duplicator-pro');
            return $result;
        } else {
            $result['successMessage'] = __("Backup Settings Saved.", 'duplicator-pro');
        }

        $sglobal->setFromInput(SnapUtil::INPUT_REQUEST);
        $sglobal->save();

        return $result;
    }

    /**
     * Save import settings
     *
     * @return array<string, mixed>
     */
    public function saveImportSettngs()
    {
        $result = ['saveSuccess' => false];
        $global = DUP_PRO_Global_Entity::getInstance();

        $global->import_chunk_size  = filter_input(
            INPUT_POST,
            'import_chunk_size',
            FILTER_VALIDATE_INT,
            array(
                'options' => array('default' => DUPLICATOR_PRO_DEFAULT_CHUNK_UPLOAD_SIZE),
            )
        );
        $global->import_custom_path = filter_input(
            INPUT_POST,
            'import_custom_path',
            FILTER_CALLBACK,
            array(
                'options' => array(
                    SnapUtil::class,
                    'sanitizeNSCharsNewlineTrim',
                ),
            )
        );
        $newRecoveryCustomPath      = filter_input(
            INPUT_POST,
            'recovery_custom_path',
            FILTER_CALLBACK,
            array(
                'options' => array(
                    SnapUtil::class,
                    'sanitizeNSCharsNewlineTrim',
                ),
            )
        );

        if (
            strlen($global->import_custom_path) > 0 &&
            (
                !is_dir($global->import_custom_path) ||
                !is_readable($global->import_custom_path)
            )
        ) {
            $result['errorMessage']     = __(
                'The custom path isn\'t a valid directory. Check that it exists or that access to it is not restricted by PHP\'s open_basedir setting.',
                'duplicator-pro'
            );
            $global->import_custom_path = '';
            $result['saveSuccess']      = false;
            return $result;
        }

        $failMessage = '';
        if ($global->setRecoveryCustomPath($newRecoveryCustomPath, $failMessage) == false) {
            $result['saveSuccess']  = false;
            $result['errorMessage'] = $failMessage;
            return $result;
        }

        if (($result['saveSuccess'] = $global->save()) == false) {
            $result['errorMessage'] = __('Can\'t save settings data', 'duplicator-pro');
        } else {
            $result['successMessage'] = __('Settings updated.', 'duplicator-pro');
        }

        return $result;
    }

    /**
     * Mysql dump message
     *
     * @param bool   $mysqlDumpFound Found
     * @param string $mysqlDumpPath  mysqldump path
     *
     * @return void
     */
    public static function getMySQLDumpMessage($mysqlDumpFound = false, $mysqlDumpPath = '')
    {
        ?>
        <?php if ($mysqlDumpFound) :
            ?>
            <span class="dup-feature-found success-color">
                <?php echo esc_html($mysqlDumpPath) ?> &nbsp;
                <small>
                    <i class="fa fa-check-circle"></i>&nbsp;<i><?php esc_html_e("Successfully Found", 'duplicator-pro'); ?></i>
                </small>
            </span>
            <?php
        else :
            ?>
            <span class="dup-feature-notfound alert-color">
                <i class="fa fa-exclamation-triangle fa-sm" aria-hidden="true"></i>
                <?php
                self::getMySqlDumpPathProblems($mysqlDumpPath, !empty($mysqlDumpPath));
                ?>
            </span>
            <?php
        endif;
    }

    /**
     * Return purge orphan Backups action URL
     *
     * @param bool $on true turn on, false turn off
     *
     * @return string
     */
    public function getTraceActionUrl($on)
    {
        $action = $this->getActionByKey(self::ACTION_GENERAL_TRACE);
        return $this->getMenuLink(
            self::L2_SLUG_GENERAL,
            null,
            array(
                'action'        => $action->getKey(),
                '_wpnonce'      => $action->getNonce(),
                '_logging_mode' => ($on ? 'on' : 'off'),
            )
        );
    }

    /**
     * Display mysql dump path problems
     *
     * @param string $path      mysqldump path
     * @param bool   $is_custom is custom path
     *
     * @return void
     */
    public static function getMySqlDumpPathProblems($path = '', $is_custom = false)
    {
        $available = DUP_PRO_DB::getMySqlDumpPath();
        $default   = false;
        if ($available) {
            if ($is_custom) {
                if (!DUP_PRO_U::isExecutable($path)) {
                    printf(
                        esc_html_x(
                            'The mysqldump program at custom path exists but is not executable. Please check file permission 
                            to resolve this problem. Please check this %1$sFAQ page%2$s for possible solution.',
                            '%1$s and %2$s are html anchor tags or link',
                            'duplicator-pro'
                        ),
                        '<a href="' . esc_url(DUPLICATOR_PRO_DUPLICATOR_DOCS_URL . 'how-to-resolve-dependency-checks') . '" target="_blank">',
                        '</a>'
                    );
                } else {
                    $default = true;
                }
            } else {
                if (!DUP_PRO_U::isExecutable($available)) {
                    printf(
                        esc_html_x(
                            'The mysqldump program at its default location exists but is not executable. 
                            Please check file permission to resolve this problem. Please check this %1$sFAQ page%2$s for possible solution.',
                            '%1$s and %2$s are html anchor tags or link',
                            'duplicator-pro'
                        ),
                        '<a href="' . esc_url(DUPLICATOR_PRO_DUPLICATOR_DOCS_URL . 'how-to-resolve-dependency-checks') . '" target="_blank">',
                        '</a>'
                    );
                } else {
                    $default = true;
                }
            }
        } else {
            if ($is_custom) {
                printf(
                    esc_html_x(
                        'The mysqldump program was not found at its custom path location. 
                        Please check is there some typo mistake or mysqldump program exists on that location. 
                        Also you can leave custom path empty to force automatic settings. If the problem persist 
                        contact your server admin for the correct path. For a list of approved providers that support mysqldump %1$sclick here%2$s.',
                        '%1$s and %2$s are html anchor tags or links',
                        'duplicator-pro'
                    ),
                    '<a href="' . esc_url(DUPLICATOR_PRO_DUPLICATOR_DOCS_URL . 'what-host-providers-are-recommended-for-duplicator/') . '" target="_blank">',
                    '</a>'
                );
            } else {
                esc_html_e(
                    'The mysqldump program was not found at its default location. 
                    To use mysqldump, ask your host to install it or for a custom mysqldump path.',
                    'duplicator-pro'
                );
            }
        }

        if ($default) {
            printf(
                esc_html_x(
                    'The mysqldump program was not found at its default location or the custom path below. 
                    Please enter a valid path where mysqldump can run. If the problem persist contact your 
                    server admin for the correct path. For a list of approved providers that support mysqldump %1$sclick here%2$s.',
                    '%1$s and %2$s are html anchor tags or links',
                    'duplicator-pro'
                ),
                '<a href="' . esc_url(DUPLICATOR_PRO_DUPLICATOR_DOCS_URL . 'what-host-providers-are-recommended-for-duplicator/') . '" target="_blank">',
                '</a>'
            );
        }
    }
}
